#############################################################################
#############################################################################
#
# latexMacros.tcl (called from latex.tcl)
#
#############################################################################
#
# Author:  Tom Scavo <trscavo@syr.edu>, Vince Darley <vince@santafe.edu>
#
#############################################################################
#############################################################################
proc latexMacros.tcl {} {}
#############################################################################
# 
# Basic Commands
#
#############################################################################
#  label returners  #
proc TeX::labelDelim {} {
	global TeXmodeVars
	return $TeXmodeVars(standardTeXLabelDelimiter)
}
proc TeX::label {type} {
    global TeXmodeVars
    return "\\label\{${type}$TeXmodeVars(standardTeXLabelDelimiter)\}"
}

#--------------------------------------------------------------------------
#  Utilities: 
#--------------------------------------------------------------------------
# A keyboard-bound method of accessing menu commands.  Takes a list of 
# menu items (i.e., the tail of a 'menu' command), the menu name (the 
# argument of the '-n' switch) , and the name of a menu filter (the 
# argument of the '-p' switch) as input, and displays these items in a 
# list box.  If the chosen item is a menu command (as opposed to a 
# submenu), it is passed to the menu filter; otherwise, 'chooseCommand' 
# recursively calls itself until a menu command is chosen or the cancel 
# button is pressed.
#
proc chooseCommand {menuItems {menuName ""} {menuFilterProc ""} {level 1}} {
	watchCursor
	if { $menuItems == "" } { return }
	# Preprocess the list of menu items:
	foreach	item $menuItems {
		regsub -all {[<!/].} $item {} item
		regsub -all {}	$item {} item
		lappend	menOut $item
		if { [string match "menu*" $item] } {
			if { [set ind [lsearch $item {-n}]] >= 0 } {
				lappend	top "[lindex $item [incr	ind]]:"
			}
		} elseif { ![string match "(*" $item] } {
			lappend	top $item
		}
	}
	# Present the menu items to the user:
	set res	[listpick -p "Choose menu command (level $level):" $top]
	# Either execute a command or recurse on a submenu:
	if { [lsearch $menOut $res] >= 0 } {
		# Execute the command via the menu filter, if necessary:
		if { $menuFilterProc == "" } {
			$res
		} else {
			$menuFilterProc $menuName $res
		}
	} else {
		set res [string trimright $res {:}]
		foreach	item $menOut {
			if { [lsearch $item $res] >= 0 } {
				set menuItems [lindex $item end]
				# Determine the name of this submenu:
				if { [set ind [lsearch $item {-n}]] >= 0 } {
					set menuName [lindex $item [incr ind]]
				} else {
					set menuName ""
				}
				# Determine the name of the menu filter for this submenu:
				if { [set ind [lsearch $item {-p}]] >= 0 } {
					set menuFilterProc [lindex $item [incr ind]]
				} else {
					set menuFilterProc ""
				}
				return [chooseCommand $menuItems $menuName $menuFilterProc [incr level]]
			}
		}
	}
}
# A keyboard-bound method of inserting environments.
#
proc chooseEnvironment {} {
	set env [getEnvironment]
	if { [catch {eval $env}] } {
		doWrapEnvironment $env
	}
}
proc getEnvironment {} {
	if { [catch {prompt "Choose environment:" "abstract" "" \
	             "abstract" \
	             "array" \
	             "center" \
	             "description" \
	             "displaymath" \
	             "document" \
	             "enumerate" \
	             "eqnarray" \
	             "eqnarray*" \
	             "equation" \
	             "figure" \
	             "filecontents" \
	             "flushleft" \
	             "flushright" \
	             "itemize" \
	             "math" \
	             "minipage" \
	             "note" \
	             "overlay" \
	             "quotation" \
	             "quote" \
	             "slide" \
	             "table" \
	             "tabular" \
	             "thebibliography" \
	             "titlepage" \
	             "verbatim" \
	             "verse" \
            	} env] } {
		set env ""
	}
	return $env
}

#############################################################################
# 
# Paragraph Mode Macros
#
#############################################################################
#--------------------------------------------------------------------------
#  Documents: 
#--------------------------------------------------------------------------
proc newLaTeXDocument {} {
	catch {prompt "Choose a documentclass:" "article" "" "article" \
	        "report" "book" "letter" "slides"} documentType
	if {$documentType != "cancel"} {
		new -m TeX
		if { [catch {${documentType}Documentclass}] } {
			wrapDocument "$documentType" 
		}
		while { [options] } {}
		message "enter option (or leave blank)"
	}
}
proc letterDocumentclass {} {
	set    preamble "\r\\address\{%\r"
	append preamble "		\\\\	% insert your name here\r"
	append preamble "		\\\\	% insert your address here\r"
	append preamble "		\\\\	% insert more address here\r"
	append preamble "		  	% insert city-state-zip here\r"
	append preamble "\}\r\r"
	append preamble "\\date\{\}  % optional\r"
	append preamble "\\signature\{\}\r\r"
	set    body     "\r\\begin\{letter\}\{%\r"
	append body     "		\\\\	% insert addressee's name here\r"
	append body     "		\\\\	% insert addressee's address here\r"
	append body     "		\\\\	% insert more address here\r"
	append body     "		  	% insert addressee's city-state-zip here\r"
	append body     "\}\r\r"
	append body     "\\opening\{Dear ,\}\r\r"
	if {[isEmptyFile]} {
		append body "% BODY OF LETTER\r"
		append body "\r\r"
	} else {
		if {[isSelectionAll]} {
			set text [getSelect]
# 			deleteText [minPos] [maxPos]
			append body "$text\r"
		} else {
			alertnote "nonempty file:  delete text or \'Select All\'\
				from the Edit menu"
			return
		}
	}
	append body "\\closing\{Sincerely,\}\r\r"
	append body "\\encl\{\}\r"
	append body "\\cc\{\}\r\r"
	append body "\\end\{letter\}\r\r"
	insertDocument "letter" $preamble $body
	message "enter option (or leave blank)"
}
proc articleDocumentclass {} {
	if { [wrapDocument "article"] } {
		message "enter option (or leave blank)"
	}
}
proc reportDocumentclass {} {
	if { [wrapDocument "report"] } {
		message "enter option (or leave blank)"
	}
}
proc bookDocumentclass {} {
	if { [wrapDocument "book"] } {
		message "enter option (or leave blank)"
	}
}
proc slidesDocumentclass {} {
	if { [wrapDocument "slides"] } {
		message "enter option (or leave blank)"
	}
}
proc otherDocumentclass {} {
	catch {prompt "What documentclass?" "article"} documentType
	if {$documentType != "cancel"} {
		if { [wrapDocument "$documentType"] } {
			message "enter option (or leave blank)"
		}
	}
}
# If an option is inserted, return true; otherwise, return false.
proc options {} {
	set option [getOption]
	if {$option != ""} {
		insertOption $option
		return 1
	}
	return 0
}
proc getOption {} {
	catch {prompt "Choose an option:" "11pt" "" "10pt" "11pt" "12pt" "(-" \
	        "letterpaper" "legalpaper" "executivepaper" "a4paper" "a5paper" \
	        "b5paper" "(-" "landscape" "(-" "final" "draft" "(-" \
	        "oneside" "twoside" "(-" "openright" "openany" "(-" \
	        "onecolumn" "twocolumn" "(-" "notitlepage" "titlepage" \
				  "openbib" "(-" "leqno" "(-" "fleqn"} optionName
	if {$optionName != "cancel"} {
		return $optionName
	} else {
		return ""
	}
}
proc insertOption {option} {
	global TeXmodeVars
	set searchString {\\documentclass}
	set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString 0]
	if {[llength $searchResult] == 0} {
		if { $TeXmodeVars(searchNoisily) } {beep}
		message "can\'t find \\documentclass"
	} else {
		set nextCharPos [lindex $searchResult 1]
		goto $nextCharPos
		set nextChar [lookAt $nextCharPos]
		if {$nextChar == "\["} {
			forwardChar
			insertText $option
			if {[lookAt [getPos]] != "\]"} {
				insertText ","
			}
		} elseif {$nextChar == "\{"} {
			insertText "\[$option\]"
		} else {
			alertnote "unrecognizable \\documentclass statement"
		}
	}
}
proc insertPackage {package} {
	global TeXmodeVars
	# Check to see if $package is already loaded:
	if { $package != "" } {
		append searchString {^[^%]*\\usepackage\{.*} $package {.*\}}
		set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString 0]
		if {[llength $searchResult] != 0} {
			if { $TeXmodeVars(searchNoisily) } {beep}
			message "$package package already loaded"
			return
		}
	}
	# Newlines are allowed in the arguments of \documentclass:
	set searchString {\\documentclass(\[[^][]*\])?{[^{}]*}}
	# Search for \documentclass command:
	set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString 0]
	if {[llength $searchResult] == 0} {
		if { $TeXmodeVars(searchNoisily) } {beep}
		message "can't find \\documentclass"
	} else {
		pushPosition
		goto [lindex $searchResult 1]
		set txt "\r\\usepackage\{$package\}"
		insertText $txt
		backwardChar
		message "Press <Ctl .> to return to previous position"
	}
}
proc filecontents {} {
	global searchNoisily
	set searchString {\\documentclass}
	set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString 0]
	if {[llength $searchResult] == 0} {
		if {$searchNoisily} {beep}
		message "can\'t find \\documentclass"
		return
	} else {
		set prompt "File to be included:"
		if {[catch {getfile $prompt} path]} {
			return
		} else {
			replaceText [minPos] [minPos] [buildFilecontents $path]
			goto [minPos]
			message "file included"
		}
	}
}
proc filecontentsAll {} {
	global searchNoisily
	watchCursor
	message "locating all input files"
	set currentWin [win::Current]
	# Is the current window part of TeX fileset?
	set fset [isWindowInFileset $currentWin "tex"]
	if { $fset == "" } {
		set searchString {\\documentclass}
		set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString 0]
		if {[llength $searchResult] == 0} {
			if {$searchNoisily} {beep}
			message "can\'t find \\documentclass"
			return
		} else {
			set text [getText [minPos] [maxPos]]
		}
	} else {
		# Will not handle a base file that is open and dirty:
		set text [buildFilecontents [texFilesetBaseName $fset]]
	}
	set currentDir [file dirname $currentWin]
	set newText [texResolveAll $text $currentDir]
	if { [string length $text] == [string length $newText] } {
		beep
		message "no files to include"
	} else {
		replaceText [minPos] [maxPos] $newText
		goto [minPos]
		message "all files included"
	}
}
# Takes a LaTeX document string and a path as input, and returns
# a modified document string with all filecontents environments
# prepended.
proc texResolveAll {latexDoc currentDir} {
	global TeXmodeVars
	set pairs [list \
	            {{\\documentclass} {.cls}} {{\\LoadClass} {.cls}} \
	            {{\\include} {.tex}} \
	            {{\\usepackage} {.sty}} {{\\RequirePackage} {.sty}} \
	            {{\\input} {}} \
	            {{\\bibliography} {.bib}} {{\\bibliographystyle} {.bst}} \
	          ]
	foreach macro $TeXmodeVars(boxMacroNames) {
		regsub {\*} $macro {\\*} macro
		lappend pairs [list \\\\$macro {}]
	}
	foreach pair $pairs {
		set cmd [lindex $pair 0]
		set ext [lindex $pair 1]
		set searchString $cmd
		append searchString {(\[[^][]*\])?{([^{}]*)}}
		set searchText $latexDoc
		while { [regexp -indices -- $searchString $searchText mtch dummy theArgs] } {
			set begPos [lindex $theArgs 0]
			set endPos [lindex $theArgs 1]
			set args [string range $searchText $begPos $endPos]
			foreach arg [split $args ,] {
				if { $cmd == {\\input} && ![string length [file extension $arg]] } {
					set ext {.tex}
				}
				set files [glob -nocomplain -path [file join $currentDir $arg] *]
				set filename [file join $currentDir $arg$ext]
				if { [lsearch -exact $files $filename] > -1 } {
					set tempDoc $latexDoc
					set latexDoc [buildFilecontents $filename]
					append latexDoc $tempDoc
				}
			}
			set searchText [string range $searchText [expr $endPos + 2] end]
		}
	}
	return $latexDoc
}
# Takes a filename as input and returns a filecontents environment 
# based on the contents of that file.  If a second argument is given,
# use that as the argument of the filecontents environment instead
# of the original filename.
proc buildFilecontents {filename {newFilename {}}} {
	set text [readFile $filename]
	# Fix end-of-line characters:
	regsub -all "\xa" $text "\xd" text
	set envName "filecontents"
	if { $newFilename == {} } {
		set envArg "{[file tail $filename]}"
	} else {
		set envArg "{$newFilename}"
	}
	return [buildEnvironment $envName $envArg "$text\r" "\r\r"]
}
#--------------------------------------------------------------------------
#  Page Layout: 
#--------------------------------------------------------------------------
proc maketitle {} {
	global searchNoisily
	set searchString {\\document(class|style)(\[.*\])?\{.*\}}
	set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString 0]
	if {[llength $searchResult] == 0} {
		if {$searchNoisily} {beep}
		message "can\'t find \\documentclass or \\documentstyle"
	} else {
		set searchPos [lindex $searchResult 1]
		set searchString {\\begin\{document\}}
		set searchResult [search -s -n -f 1 -m 0 -i 1 -r 1 $searchString $searchPos]
		if {[llength $searchResult] == 0} {
			if {$searchNoisily} {beep}
			message "can\'t find \\begin\{document\}"
		} else {
			goto [lindex $searchResult 1]
			set currentPos [getPos]
			set txt "\r\r% Definition of title page:"
			append txt "\r\\title\{"
			append txt "\r\ttitle\r\}"
			append txt "\r\\author\{"
			append txt "\r\t\t% insert author(s) here"
			append txt "\r\}"
			append txt "\r\\date\{\}\t% optional"
			append txt "\r\r\\maketitle"
			elec::Insertion $txt
		}
	}
}
proc abstract {} { doWrapEnvironment "abstract" }
proc titlepage {} { doWrapEnvironment "titlepage" }
proc getPagestyle {} {
	catch {prompt "Choose a pagestyle:" "plain" "" "plain" "empty" \
				  "headings" "myheadings"} pagestyleName
	if {$pagestyleName != "cancel"} {
		return $pagestyleName
	} else {
		return ""
	}
}
proc pagestyle {} {
    set pagestyleName [getPagestyle]
    if {$pagestyleName != ""} {
	insertObject "[openingCarriageReturn]\\pagestyle\{$pagestyleName\}[closingCarriageReturn]"
    }
}
proc thispagestyle {} {
    set pagestyleName [getPagestyle]
    if {$pagestyleName != ""} {
	insertObject "[openingCarriageReturn]\\thispagestyle\{$pagestyleName\}[closingCarriageReturn]"
    }
}

proc pagenumbering {} {
    set pagenumberingStyle [getPagenumberingStyle]
    if {$pagenumberingStyle != ""} {
	insertObject "[openingCarriageReturn]\\pagenumbering\{$pagenumberingStyle\}[closingCarriageReturn]"
    }
}
proc getPagenumberingStyle {} {
    catch {prompt "Choose a pagenumbering style:" "arabic" "" "arabic" \
      "roman" "Roman" "alph" "Alph"} pagenumberingStyle
    if {$pagenumberingStyle != "cancel"} {
	return $pagenumberingStyle
    } else {
	return ""
    }
}
proc twocolumn {} {
    insertObject "[openingCarriageReturn]\\twocolumn[closingCarriageReturn]"
}
proc onecolumn {} {
    insertObject "[openingCarriageReturn]\\onecolumn[closingCarriageReturn]"
}
#--------------------------------------------------------------------------
#  Sectioning: 
#--------------------------------------------------------------------------
proc sectioning {item} {
	append left [openingCarriageReturn] "\\${item}\{"
	append right "\}"
	if [regexp (part|chapter) $item] {
		append right "\\thispagestyle{empty}"
	}
	append right "" [closingCarriageReturn]
	if [elec::Wrap $left $right] {
		message "don't forget the label <Ctl Opt L>"
	} else {
		message "type the [string trim $item {*}] name and don't forget the label <Ctl Opt L>"
	}
}
proc appendix {} {
	insertObject "[openingCarriageReturn]\\appendix[closingCarriageReturn]"
}
#--------------------------------------------------------------------------
#  Text Style: 
#--------------------------------------------------------------------------

proc emph {} {
	if {[elec::Wrap "\\emph{" "}"]} {
		message "selected text has been emphasized"
	} else {
		message "enter text to be emphasized"
	}
}
proc textup {} {
	if {[elec::Wrap "\\textup{" "}"]} {
		message "selected text has upright shape"
	} else {
		message "enter text to have upright shape"
	}
}
proc textit {} {
	if {[elec::Wrap "\\textit{" "}"]} {
		message "selected text has italic shape"
	} else {
		message "enter text to have italic shape"
	}
}
proc textsl {} {
	if {[elec::Wrap "\\textsl{" "}"]} {
		message "selected text has slanted shape"
	} else {
		message "enter text to have slanted shape"
	}
}
proc textsc {} {
	if {[elec::Wrap "\\textsc{" "}"]} {
		message "selected text has small caps shape"
	} else {
		message "enter text to have small caps shape"
	}
}
proc textmd {} {
	if {[elec::Wrap "\\textmd{" "}"]} {
		message "selected text has been set in medium series"
	} else {
		message "enter text to be set in medium series"
	}
}
proc textbf {} {
	if {[elec::Wrap "\\textbf{" "}"]} {
		message "selected text has been set in bold series"
	} else {
		message "enter text to be set in bold series"
	}
}
proc textrm {} {
	if {[elec::Wrap "\\textrm{" "}"]} {
		message "selected text has been set with roman family"
	} else {
		message "enter text to be set using roman family"
	}
}
proc textsf {} {
	if {[elec::Wrap "\\textsf{" "}"]} {
		message "selected text has been set with sans serif family"
	} else {
		message "enter text to be set using sans serif family"
	}
}
proc texttt {} {
	if {[elec::Wrap "\\texttt{" "}"]} {
		message "selected text has been set with typewriter family"
	} else {
		message "enter text to be set using typewriter family"
	}
}
proc textnormal {} {
	if {[elec::Wrap "\\textnormal{" "}"]} {
		message "selected text has been set with normal style"
	} else {
		message "enter text to be set using normal style"
	}
}

proc em {} {
	if {[elec::Wrap "{\\em " "}"]} {
		message "emphasized text set"
	} else {
		message "enter text to be emphasized"
	}
}
proc upshape {} {
	if {[elec::Wrap "{\\upshape " "}"]} {
		message "text set in upright shape"
	} else {
		message "enter text to be set in upright shape"
	}
}
proc itshape {} {
	if {[elec::Wrap "{\\itshape " "}"]} {
		message "text set in italics shape"
	} else {
		message "enter text to be set in italics shape"
	}
}
proc slshape {} {
	if {[elec::Wrap "{\\slshape " "}"]} {
		message "text set in slanted shape"
	} else {
		message "enter text to be set in slanted shape"
	}
}
proc scshape {} {
	if {[elec::Wrap "{\\scshape " "}"]} {
		message "text set in small caps shape"
	} else {
		message "enter text to be set in small caps shape"
	}
}
proc mdseries {} {
	if {[elec::Wrap "{\\mdseries " "}"]} {
		message "text set in medium series"
	} else {
		message "enter text to be set in medium series"
	}
}
proc bfseries {} {
	if {[elec::Wrap "{\\bfseries " "}"]} {
		message "text set in bold series"
	} else {
		message "enter text to be set in bold series"
	}
}
proc rmfamily {} {
	if {[elec::Wrap "{\\rmfamily " "}"]} {
		message "text set in roman family"
	} else {
		message "enter text to be set in roman family"
	}
}
proc sffamily {} {
	if {[elec::Wrap "{\\sffamily " "}"]} {
		message "text set in sans serif family"
	} else {
		message "enter text to be set in sans serif family"
	}
}
proc ttfamily {} {
	if {[elec::Wrap "{\\ttfamily " "}"]} {
		message "text set in typewriter family"
	} else {
		message "enter text to be set in typewriter family"
	}
}
proc normalfont {} {
	if {[elec::Wrap "{\\normalfont " "}"]} {
		message "text set in normal style"
	} else {
		message "enter text to be set in normal style"
	}
}

#--------------------------------------------------------------------------
#  Text Size: 
#--------------------------------------------------------------------------

proc doTextSize {textSize} {
	if {[elec::Wrap "{\\$textSize " "}"]} {
		message "$textSize text set"
	} else {
		message "enter $textSize text"
	}
}

#--------------------------------------------------------------------------
#  Text Commands: 
#--------------------------------------------------------------------------

proc textsuperscript {} {
	if {[elec::Wrap "\\textsuperscript{" "}"]} {
		message "text superscripted"
	} else {
		message "enter superscripted text"
	}
}
proc textcircled {} {
	if {[elec::Wrap "\\textcircled{" "}"]} {
		message "text circled"
	} else {
		message "enter circled text"
	}
}

#--------------------------------------------------------------------------
#  International: 
#--------------------------------------------------------------------------

proc {} {} {
	if {[elec::Wrap "\\`{" "}"]} {
		message "accent set"
	} else {
		message "enter single character"
	}
}
proc {} {} {
	if {[elec::Wrap "\\'{" "}"]} {
		message "accent set"
	} else {
		message "enter single character"
	}
}
proc {} {} {
	if {[elec::Wrap "\\^{" "}"]} {
		message "accent set"
	} else {
		message "enter single character"
	}
}
proc {} {} {
	if {[elec::Wrap "\\\"{" "}"]} {
		message "accent set"
	} else {
		message "enter single character"
	}
}
proc {} {} {
	if {[elec::Wrap "\\~{" "}"]} {
		message "accent set"
	} else {
		message "enter single character"
	}
}
proc {} {} {insertObject "\\c\{c\}"}
proc {} {} {insertObject "\\c\{C\}"}
proc {} {} {insertObject "\\oe"}
proc {} {} {insertObject "\\OE"}
proc {} {} {insertObject "\\ae"}
proc {} {} {insertObject "\\AE"}
proc {} {} {insertObject "\\aa"}
proc {} {} {insertObject "\\AA"}
proc {} {} {insertObject "\\o"}
proc {} {} {insertObject "\\O"}
proc {ss} {} {insertObject "\\ss"}
proc {SS} {} {insertObject "\\SS"}
proc {} {} {insertObject "?`"}
proc {} {} {insertObject "!`"}


#--------------------------------------------------------------------------
#  Boxes: 
#--------------------------------------------------------------------------

proc mbox {} {
	if {[elec::Wrap "\\mbox{" "}"]} {
		message "mbox set"
	} else {
		message "enter text"
	}
}
proc makebox {} {
	if {[elec::Wrap "\\makebox\[\]\[\]{" "}"]} {
		message "makebox set; enter the width and position"
	} else {
		message "enter the width and position of the makebox, then the text"
	}
}
proc fbox {} {
	if {[elec::Wrap "\\fbox{" "}"]} {
		message "fbox set"
	} else {
		message "enter text"
	}
}
proc framebox {} {
	if {[elec::Wrap "\\framebox\[\]\[\]{" "}"]} {
		message "framebox set; enter the width and position"
	} else {
		message "enter the width and position of the framebox, then the text"
	}
}
proc newsavebox {} {
	if {[elec::Wrap "\\newsavebox{" "}"]} {
		message "newsavebox defined"
	} else {
		message "enter the command name of the sbox or savebox"
	}
}
proc sbox {} {
	if {[elec::Wrap "\\sbox{}{" "}"]} {
		message "sbox set; enter the command name"
	} else {
		message "enter the command name of the sbox, then the text"
	}
}
proc savebox {} {
	if {[elec::Wrap "\\savebox{}\[\]\[\]{" "}"]} {
		message "savebox set; enter the command name"
	} else {
		message "enter the command name of the savebox"
	}
}
proc usebox {} {
	if {[elec::Wrap "\\usebox{" "}"]} {
		message "usebox declared"
	} else {
		message "enter the command name of the sbox or savebox"
	}
}
proc raisebox {} {
	if {[elec::Wrap "\\raisebox{}\[\]\[\]{" "}"]} {
		message "raisebox set; enter the displacement"
	} else {
		message "enter the displacement of the raisebox"
	}
}
proc parbox {} {
	if {[elec::Wrap "\\parbox\[\]\{\}{" "}"]} {
		message "parbox set; enter the position and width"
	} else {
		message "enter the position \[b|c|t\] and width of the parbox, then the text"
	}
}
proc rule {} {
	insertObject "\\rule\[\]\{\}{}"
	message "enter the displacement of the rule, then width and height"
}

#--------------------------------------------------------------------------
#  Misc: 
#--------------------------------------------------------------------------

proc verb {} {
	if {[elec::Wrap "\\verb|" "|"]} {
		message "verbatim text set"
	} else {
		message "enter verbatim text"
	}
}
proc footnote {} {
	if {[elec::Wrap "\\footnote{" "}"]} {
		message "footnote set"
	} else {
		message "enter footnote"
	}
}
proc marginalNote {} {
	if {[elec::Wrap "\\marginpar{" "}"]} {
		message "marginal note set"
	} else {
		message "enter marginal note"
	}
}

proc insertLabel {} {
	if {[elec::Wrap "\\label{" "}"]} {
		message "label defined"
	} else {
		message "enter label"
	}
}
proc ref {} { 
    global completion::in_progress_pos
    if {[pos::compare [set completion::in_progress_pos] == [getPos]]} {
	bind::Completion
    } else {
	if {[elec::Wrap "\\ref{" "}" 1]} {
		message "reference made"
	} else {
		message "enter reference label"
	}
    }
}
proc eqref {} { 
    global completion::in_progress_pos standardTeXLabelDelimiter
    if {[pos::compare [set completion::in_progress_pos] == [getPos]]} {
	bind::Completion
    } else {
	if {[elec::Wrap "\\eqref\{eq${standardTeXLabelDelimiter}" "\}" 1]} {
		message "reference made"
	} else {
		message "enter reference label"
	}
    }
}
proc pageref {} { 
    global completion::in_progress_pos
    if {[pos::compare [set completion::in_progress_pos] == [getPos]]} {
	bind::Completion
    } else {
	if {[elec::Wrap "\\pageref{" "}" 1]} {
		message "page reference made"
	} else {
		message "enter page reference label"
	}
    }
}
proc cite {} {
	if {[elec::Wrap "\\cite{" "}" 1]} {
		message "citation made"
	} else {
		message "enter citation key"
	}
}
proc nocite {} {
	if {[elec::Wrap "\\nocite{" "}"]} {
		message "citation added to the list"
	} else {
		message "enter citation key"
	}
}

# Insert an \item or a \bibitem, depending on the context.
proc insertItem {} {
	set command [eval getText [searchEnvironment]]
	set environment [extractCommandArg $command]
	switch $environment {
		"itemize" {
			set text "\\item  "
		}
		"enumerate" {
			set text "\\item  "
		}
		"description" {
			set text "\\item\[\]  "
		}
		"thebibliography" {
			set text "\\bibitem{}  "
		}
		default {
			beep
			message "insertItem: cursor in $environment environment"
			return
		}
	}
	set pos [getPos]
	# Indentation should mirror that of an existing \item
	# (if it exists)
	elec::Insertion [openingCarriageReturn]$text
}

proc quotes {} {
	if {[elec::Wrap "`" "'"]} {
		message "text quoted"
	} else {
		message "enter text"
	}
}
proc dblQuotes {} {
	if {[elec::Wrap "``" "''"]} {
		message "text double quoted"
	} else {
		message "enter text"
	}
}

proc texLogo {} {insertObject "\\TeX"}
proc latexLogo {} {insertObject "\\LaTeX"}
proc latex2eLogo {} {insertObject "\\LaTeXe"}
proc today {} {insertObject "\\today"}

proc dag {} {insertObject "\\dag"}
proc ddag {} {insertObject "\\ddag"}
proc sectionMark {} {insertObject "\\S"}
proc paragraphMark {} {insertObject "\\P"}
proc copyright {} {insertObject "\\copyright"}
proc pounds {} {insertObject "\\pounds"}


#############################################################################
# 
# Math Mode Macros
#
#############################################################################

#--------------------------------------------------------------------------
#  Math Modes: 
#--------------------------------------------------------------------------

proc texMath {} {
	checkMathMode "texMath" 0
	if {[elec::Wrap "$" "$"]} {
		message "formula set"
	} else {
		message "enter formula"
	}
}
proc texDisplaymath {} {
	checkMathMode "texDisplaymath" 0
	if {[elec::Wrap "$$" "$$"]} {
		message "displayed formula set"
	} else {
		message "enter displayed formula"
	}
}
proc latexMath {} {
	checkMathMode "latexMath" 0
	if {[elec::Wrap "\\( " " \\)"]} {
		message "formula set"
	} else {
		message "enter formula"
	}
}
proc latexDisplaymath {} {
	checkMathMode "latexDisplaymath" 0
	if {[elec::Wrap "\\\[ " " \\\]"]} {
		message "displayed formula set"
	} else {
		message "enter displayed formula"
	}
}

#--------------------------------------------------------------------------
#  Math Style: 
#--------------------------------------------------------------------------

proc doMathStyle {mathStyle description} {
	checkMathMode "$mathStyle" 1
	if {[elec::Wrap "\\$mathStyle{" "}"]} {
		message "selected text is $description"
	} else {
		message "enter text to be $description"
	}
}
proc doUppercaseMathStyle {mathStyle description} {
	checkMathMode "$mathStyle" 1
	# Allow upper-case alphabetic arguments only:
	if {[isSelection] && (![isUppercase] || ![isAlphabetic])} {
		beep
		alertnote "argument to \\$mathStyle must be UPPERCASE alphabetic"
		return
	}
	if {[elec::Wrap "\\$mathStyle{" "}"]} {
		message "selected text is $description"
	} else {
		message "enter text to be $description (UPPERCASE letters only)"
	}
}


#--------------------------------------------------------------------------
#  Formulas: 
#--------------------------------------------------------------------------

proc subscript {} {
	checkMathMode "subscript" 1
	if {[elec::Wrap "_{" "}"]} {
		message "subscript set"
	} else {
		message "enter subscript"
	}
}
proc superscript {} {
	checkMathMode "superscript" 1
	if {[elec::Wrap "^{" "}"]} {
		message "superscript set"
	} else {
		message "enter superscript"
	}
}
proc fraction {} {
	checkMathMode "fraction" 1
	set currentPos [getPos]
	if {[isSelection]} {
		set selection [getSelect]
		set args [split $selection /]
		set len [llength $args]
		deleteText $currentPos [selEnd]
		if {$len == 1} {
			insertObject "\\frac{$selection}{}"
			message "enter denominator"
		} else {
			set firstArg [lindex $args 0]
			set restArgs [lrange $args 1 [expr $len-1]]
			insertObject "\\frac{$firstArg}{[join $restArgs /]}"
			if {$len > 2} {message "beware of multiple /"}
		}
	} else {
		insertObject "\\frac{}{}"
		message "enter numerator"
	}
}
proc squareRoot {} {
	checkMathMode "squareRoot" 1
	if {[elec::Wrap "\\sqrt{" "}"]} {
		message "square root set"
	} else {
		message "enter formula"
	}
}
proc nthRoot {} {
	checkMathMode "nthRoot" 1
	if {[elec::Wrap "\\sqrt\[\]{" "}"]} {
		message "enter root"
	} else {
		message "enter root, then formula"
	}
}
proc oneParameter {} {
	checkMathMode "oneParameter" 1
	if {[elec::Wrap "\\{" "}"]} {
		message "enter command name"
	} else {
		message "enter command name, press <Tab>, enter argument"
	}
}
proc twoParameters {} {
	checkMathMode "twoParameters" 1
	if {[elec::Wrap "\\{" "}{}"]} {
		message "enter command name"
	} else {
		message "enter command name, press <Tab>, enter argument, etc."
	}
}

#--------------------------------------------------------------------------
#  Large Ops: 
#--------------------------------------------------------------------------

proc insertLargeOp {commandName} {
	checkMathMode "$commandName" 1
	insertObject "\\$commandName\_{}^{}"
}
proc sum {} {insertLargeOp "sum"}
proc prod {} {insertLargeOp "prod"}
proc coprod {} {insertLargeOp "coprod"}
proc int {} {insertLargeOp "int"}
proc oint {} {insertLargeOp "oint"}
proc bigcap {} {insertLargeOp "bigcap"}
proc bigcup {} {insertLargeOp "bigcup"}
proc bigsqcup {} {insertLargeOp "bigsqcup"}
proc bigvee {} {insertLargeOp "bigvee"}
proc bigwedge {} {insertLargeOp "bigwedge"}
proc bigodot {} {insertLargeOp "bigodot"}
proc bigotimes {} {insertLargeOp "bigotimes"}
proc bigoplus {} {insertLargeOp "bigoplus"}
proc biguplus {} {insertLargeOp "biguplus"}

#--------------------------------------------------------------------------
#  Delimiters: 
#--------------------------------------------------------------------------

proc delimitObject {leftDelim rightDelim} {
	if {[elec::Wrap $leftDelim $rightDelim]} {
		message "formula delimited"
	} else {
		message "enter formula"
	}
}
proc parentheses {} { checkMathMode "parentheses" 1; delimitObject "(" ")" }
proc brackets {} { checkMathMode "brackets" 1; delimitObject "\[" "\]" }
proc braces {} { checkMathMode "braces" 1; delimitObject "\\\{" "\\\}" }
proc absoluteValue {} { checkMathMode "absoluteValue" 1; delimitObject "|" "|" }
proc getDelims {} {
	catch {prompt "Choose delimiters:" "parentheses" "" "parentheses" \
				  "brackets" "braces" "angle brackets" "vertical bars" \
				  "double bars" "ceiling" "floor"} delimType
	if {$delimType != "cancel"} {
		switch $delimType {
			"parentheses" {
				set leftDelim "("
				set rightDelim ")"
			}
			"brackets" {
				set leftDelim "\["
				set rightDelim "\]"
			}
			"braces" {
				set leftDelim "\\\{"
				set rightDelim "\\\}"
			}
			"vertical bars" {
				set leftDelim "|"
				set rightDelim "|"
			}
			"double bars" {
				set leftDelim "\\|"
				set rightDelim "\\|"
			}
			"angle brackets" {
				set leftDelim "\\langle"
				set rightDelim "\\rangle"
			}
			"ceiling" {
				set leftDelim "\\lceil"
				set rightDelim "\\rceil"
			}
			"floor" {
				set leftDelim "\\lfloor"
				set rightDelim "\\rfloor"
			}
			default {
				alertnote "\"$delimType\" not recognized"
				return ""
			}
		}
		return [list $leftDelim $rightDelim]
	} else {return ""}
}
proc otherDelims {} {
	checkMathMode "otherDelims" 1
	set delims [getDelims]
	if {$delims != ""} {
		set leftDelim [lindex $delims 0]
		set rightDelim [lindex $delims 1]
		delimitObject "$leftDelim" "$rightDelim"
	}
}
proc {half-openInterval} {} {
	checkMathMode "half-openInterval" 1; delimitObject "(" "\]"
}
proc {half-closedInterval} {} {
	checkMathMode "half-closedInterval" 1; delimitObject "\[" ")"
}
proc insertBigDelims {leftDelim rightDelim isMultiline} {
	checkMathMode "insertBigDelims" 1
	if {$isMultiline} {
		doWrapStructure $leftDelim "" $rightDelim
	} else {
		if { [elec::Wrap $leftDelim $rightDelim] } {
			message "formula delimited"
		} else {
			message "enter formula"
		}
	}
}
proc bigParens {} {
	checkMathMode "bigParens" 1; insertBigDelims "\\left(" "\\right)" 0
}
proc multiBigParens {} {
	checkMathMode "multiBigParens" 1; insertBigDelims "\\left(" "\\right)" 1
}
proc bigBrackets {} {
	checkMathMode "bigBrackets" 1; insertBigDelims "\\left\[" "\\right\]" 0
}
proc multiBigBrackets {} {
	checkMathMode "multiBigBrackets" 1; insertBigDelims "\\left\[" "\\right\]" 1
}
proc bigBraces {} {
	checkMathMode "bigBraces" 1; insertBigDelims "\\left\\\{" "\\right\\\}" 0
}
proc multiBigBraces {} {
	checkMathMode "multiBigBraces" 1; insertBigDelims "\\left\\\{" "\\right\\\}" 1
}
proc bigAbsValue {} {
	checkMathMode "bigAbsValue" 1; insertBigDelims "\\left|" "\\right|" 0
}
proc multiBigAbsValue {} {
	checkMathMode "multiBigAbsValue" 1; insertBigDelims "\\left|" "\\right|" 1
}
proc doOtherBigDelims {name isMultiline} {
	checkMathMode $name 1
	set delims [getDelims]
	if {$delims != ""} {
		append leftDelim "\\left" [lindex $delims 0]
		append rightDelim "\\right" [lindex $delims 1]
		insertBigDelims "$leftDelim" "$rightDelim" $isMultiline
	}
}
proc otherBigDelims {} {
	doOtherBigDelims "otherBigDelims" 0
}
proc otherMultiBigDelims {} {
	doOtherBigDelims "otherMultiBigDelims" 1
}
proc bigLeftBrace {} {
	checkMathMode "bigLeftBrace" 1
	insertBigDelims "\\left\\\{" "\\right." 0
}
proc multiBigLeftBrace {} {
	checkMathMode "multiBigLeftBrace" 1
	insertBigDelims "\\left\\\{" "\\right." 1
}
proc doOtherMixedBigDelims {name isMultiline} {
	checkMathMode $name 1
	catch {prompt "Choose LEFT delimiter:" "parenthesis" "" "parenthesis" \
	              "bracket" "brace" "vertical bar" "double bar" \
	              "angle bracket" "ceiling" "floor" "slash" "backslash" \
	              "none"} delimType
	if {$delimType != "cancel"} {
		switch $delimType {
			"parenthesis" {set leftDelim "("}
			"bracket" {set leftDelim "\["}
			"brace" {set leftDelim "\\\{"}
			"vertical bar" {set leftDelim "|"}
			"double bar" {set leftDelim "\\|"}
			"angle bracket" {set leftDelim "\\langle"}
			"ceiling" {set leftDelim "\\lceil"}
			"floor" {set leftDelim "\\lfloor"}
			"slash" {set leftDelim "/"}
			"backslash" {set leftDelim "\\backslash"}
			"none" {set leftDelim "."}
			default {
				alertnote "\"$delimType\" not recognized"
				return
			}
		}
		catch {prompt "Choose RIGHT delimiter:" "parenthesis" "" "parenthesis" \
		              "bracket" "brace" "vertical bar" "double bar" \
		              "angle bracket" "ceiling" "floor" "slash" "backslash" \
		              "none"} delimType
		if {$delimType != "cancel"} {
			switch $delimType {
				"parenthesis" {set rightDelim ")"}
				"bracket" {set rightDelim "\]"}
				"brace" {set rightDelim "\\\}"}
				"vertical bar" {set rightDelim "|"}
				"double bar" {set rightDelim "\\|"}
				"angle bracket" {set rightDelim "\\rangle"}
				"ceiling" {set rightDelim "\\rceil"}
				"floor" {set rightDelim "\\rfloor"}
				"slash" {set rightDelim "/"}
				"backslash" {set rightDelim "\\backslash"}
				"none" {set rightDelim "."}
				default {
					alertnote "\"$delimType\" not recognized"
					return
				}
			}
			insertBigDelims "\\left$leftDelim" "\\right$rightDelim" $isMultiline
		}
	}
}
proc otherMixedBigDelims {} {
	doOtherMixedBigDelims "otherMixedBigDelims" 0
}
proc otherMultiMixedBigDelims {} {
	doOtherMixedBigDelims "otherMultiMixedBigDelims" 1
}
#--------------------------------------------------------------------------
#  Accents: 
#--------------------------------------------------------------------------
proc acute {} {
	checkMathMode "acute" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\acute{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc bar {} {
	checkMathMode "bar" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\bar{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc breve {} {
	checkMathMode "breve" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\breve{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc check {} {
	checkMathMode "check" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\check{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc dot {} {
	checkMathMode "dot" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\dot{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc ddot {} {
	checkMathMode "ddot" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\ddot{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc grave {} {
	checkMathMode "grave" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\grave{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc hat {} {
	checkMathMode "hat" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\hat{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc tilde {} {
	checkMathMode "tilde" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\tilde{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc vec {} {
	checkMathMode "vec" 1
	if {[isSelection] > 1} {
		alertnote "Warning: only a single character may be accented!"
	}
	if {[elec::Wrap "\\vec{" "}"]} {
		message "accent set"
	} else {
		message "enter one character"
	}
}
proc widehat {} {
	checkMathMode "widehat" 1
	if {[isSelection] > 3} {
		alertnote "Warning: only a few characters may be accented!"
	}
	if {[elec::Wrap "\\widehat{" "}"]} {
		message "accent set"
	} else {
		message "enter a few characters"
	}
}
proc widetilde {} {
	checkMathMode "widetilde" 1
	if {[isSelection] > 3} {
		alertnote "Warning: only a few characters may be accented!"
	}
	if {[elec::Wrap "\\widetilde{" "}"]} {
		message "accent set"
	} else {
		message "enter a few characters"
	}
}

#--------------------------------------------------------------------------
#  Grouping: 
#--------------------------------------------------------------------------
proc underline {} {
	checkMathMode "underline" 1
	if {[elec::Wrap "\\underline{" "}"]} {
		message "selection underlined"
	} else {
		message "enter text"
	}
}
proc overline {} {
	checkMathMode "overline" 1
	if {[elec::Wrap "\\overline{" "}"]} {
		message "selection overlined"
	} else {
		message "enter text"
	}
}
proc underbrace {} {
	checkMathMode "underbrace" 1
	if {[elec::Wrap "\\underbrace{" "}"]} {
		message "selection underbraced"
	} else {
		message "enter text"
	}
}
proc overbrace {} {
	checkMathMode "overbrace" 1
	if {[elec::Wrap "\\overbrace{" "}"]} {
		message "selection overbraced"
	} else {
		message "enter text"
	}
}
proc overrightarrow {} {
	checkMathMode "overrightarrow" 1
	if {[elec::Wrap "\\overrightarrow{" "}"]} {
		message "selection overrightarrowed"
	} else {
		message "enter text"
	}
}
proc overleftarrow {} {
	checkMathMode "overleftarrow" 1
	if {[elec::Wrap "\\overleftarrow{" "}"]} {
		message "selection overleftarrowed"
	} else {
		message "enter text"
	}
}
proc stackrel {} {
	checkMathMode "stackrel" 1
	set currentPos [getPos]
	if {[insertObject "\\stackrel{}{}"]} {
		message "1st arg scriptstyle"
	}
}
#--------------------------------------------------------------------------
#  Spacing: 
#--------------------------------------------------------------------------
proc negThin {} {checkMathMode "negThin" 1; insertObject "\\!"}
proc thin {} {checkMathMode "thin" 1; insertObject "\\,"}
proc medium {} {checkMathMode "medium" 1; insertObject "\\:"}
proc thick {} {checkMathMode "thick" 1; insertObject "\\;"}
proc quad {} {checkMathMode "quad" 1; insertObject "\\quad"}
proc qquad {} {checkMathMode "qquad" 1; insertObject "\\qquad"}
proc hspace {} {
	checkMathMode "hspace" 1
	if {[elec::Wrap "\\hspace{" "}"]} {
		message "spacing set"
	} else {
		message "enter the desired horizontal spacing"
	}
}
proc vspace {} {
	checkMathMode "vspace" 1
	if {[elec::Wrap "\\vspace{" "}"]} {
		message "spacing set"
	} else {
		message "enter the desired horizontal spacing"
	}
}
proc hfill {} {checkMathMode "hfill" 1; insertObject "\\hfill"}
proc vfill {} {checkMathMode "vfill" 1; insertObject "\\vfill"}
proc smallskip {} {checkMathMode "smallskip" 1; insertObject "\\smallskip"}
proc medskip {} {checkMathMode "medskip" 1; insertObject "\\medskip"}
proc bigskip {} {checkMathMode "bigskip" 1; insertObject "\\bigskip"}
